#ifndef AUBO_SCOPE_PERSISTED_VARIABLE_H
#define AUBO_SCOPE_PERSISTED_VARIABLE_H

#include <aubo_caps/domain/variable/variable.h>

namespace arcs {
namespace aubo_scope {
ARCS_CLASS_FORWARD(PersistedVariable);

/**
 * This interface represents variables stored with the installation. These
 * variables are defined in the installation and will exist even when a robot is
 * turned off and on.
 */
class ARCS_ABI_EXPORT PersistedVariable : public Variable
{
public:
    PersistedVariable(PersistedVariable &f);
    PersistedVariable(PersistedVariable &&f);
    virtual ~PersistedVariable();

    /**
     * A persisted variable can not be guaranteed to be present. This method can
     * be used to determine, if the variable is present.
     *
     *The persisted variable will not be present, if the end user loads a
     *different installation which does not contain the variable, or if the
     *variable is deleted by the end user.
     *
     *  A program node storing a <code>PersistedVariable</code> in its Data
     *Model will be undefined if the variable cannot be resolved (i.e. the
     *variable is not present).
     *
     * @return <code>true</code>, if this persisted variable is resolvable (i.e.
     * present in the current installation).
     */
    bool isResolvable();

protected:
    PersistedVariable();

private:
    friend class DataSwitch;
    void *d_{ nullptr };
};

struct PersistedVariableFilter
{
    bool operator()(VariablePtr var)
    {
        return std::dynamic_pointer_cast<PersistedVariable>(var) != nullptr &&
               var->getType() == Variable::VALUE_PERSISTED;
    }
};

} // namespace aubo_scope
} // namespace arcs
#endif // AUBO_SCOPE_PERSISTED_VARIABLE_H
