#ifndef AUBO_SCOPE_VARIABLE_FACTORY_H
#define AUBO_SCOPE_VARIABLE_FACTORY_H

#include <aubo_caps/domain/value/expression/expression.h>
#include <aubo_caps/domain/variable/global_variable.h>
#include <aubo_caps/domain/variable/feature_variable.h>

namespace arcs {
namespace aubo_scope {
ARCS_CLASS_FORWARD(VariableFactory);

/**
 * <p>
 * This interface supplies methods to create variable objects.
 * </p>
 *
 * <b>NOTE:</b> Only use this functionality in a AuboCap program node (not
 * relevant for an installation node).
 */
class ARCS_ABI_EXPORT VariableFactory
{
public:
    VariableFactory(VariableFactory &f);
    VariableFactory(VariableFactory &&f);
    ~VariableFactory();

    /**
     * <p>
     * Creates a global variable. Registering the variable in AuboScope (by
     * storing it in a {@link DataModel} instance or using it for the
     * configuration of a built-in AuboScope program node) will make it
     * accessible to all program nodes.
     * </p>
     *
     * <b>NOTE:</b> Only use this method in a AuboCap program node (not relevant
     * for an installation node).
     *
     * @param suggested_name Suggested name of the variable. Valid names must
     * match regex [a-zA-Z][a-zA-Z0-9_]{0,14} for a total of 15 characters.
     *
     * @return GlobalVariable.
     * @throws VariableException In case of an illegal name.
     */
    GlobalVariablePtr createGlobalVariable(const std::string &suggested_name);

    /**
     * <p>
     * Creates a global variable.
     * </p>
     *
     * <p>
     * Registering the variable in AuboScope (by storing it in a {@link
     * DataModel} instance or using it for the configuration of a built-in
     * AuboScope program node) will make it accessible to all program nodes.
     * </p>
     *
     * <p>
     * Setting an initial value will make it possible to read from the variable
     * immediately without having to assign a value to it first. This can be
     * useful in various constructs which requires an initial value, e.g. 'var_1
     * = var_1 + 1' or 'if (var_2 =):'.
     * </p>
     *
     * <b>NOTE:</b> Only use this method in a AuboCap program node (not relevant
     * for an installation node).
     *
     * @param suggested_name Suggested name of the variable. Valid names must
     * match regex [a-zA-Z][a-zA-Z0-9_]{0,14} for a total of 15 characters.
     * @param initialValue The expression that will be evaluated in order to
     * initialize the value of the variable. This value is assigned before the
     * actual program runs.
     *
     * @return GlobalVariable.
     * @throws VariableException In case of an illegal name.
     */
    GlobalVariablePtr createGlobalVariable(const std::string &suggested_name,
                                           ExpressionPtr initialValue);

    /**
     * <p>
     * Creates a feature variable.
     * </p>
     *
     * <p>
     * Registering the variable in AuboScope (by storing it in a {@link
     * DataModel} instance or using it for the configuration of a built-in
     * AuboScope program node) will make it accessible to all program nodes.
     * </p>
     *
     * <p>
     * Setting an initial value will make it possible to read from the variable
     * immediately without having to assign a value to it first. This can be
     * useful in various constructs which requires an initial valu.
     * </p>
     *
     * <b>NOTE:</b> Only use this method in a AuboCap program node (not relevant
     * for an installation node).
     *
     * @param suggested_name Suggested name of the variable. Valid names must
     * match regex [a-zA-Z][a-zA-Z0-9_]{0,14} for a total of 15 characters.
     * @param initialValue The expression that will be evaluated in order to
     * initialize the value of the variable. This value is assigned before the
     * actual program runs.
     *
     * @return FeatureVariable.
     * @throws VariableException In case of an illegal name.
     */
    FeatureVariablePtr createFeatureVariable(const std::string &suggested_name,
                                             const std::vector<double> &pose);

private:
    friend class DataSwitch;
    VariableFactory();
    void *d_{ nullptr };
};

} // namespace aubo_scope
} // namespace arcs
#endif // AUBO_SCOPE_VARIABLE_FACTORY_H
