#ifndef AUBO_SCOPE_VARIABLE_MODEL_H
#define AUBO_SCOPE_VARIABLE_MODEL_H

#include <vector>
#include <functional>
#include <aubo_caps/domain/feature/feature.h>
#include <aubo_caps/domain/variable/variable_factory.h>
#include <aubo_caps/domain/variable/feature_variable.h>

namespace arcs {
namespace aubo_scope {
ARCS_CLASS_FORWARD(VariableModel);

/**
 * Provides methods that returns the variables that currently are registered in
 * the system. A variable is registered if it has been stored in a {@link
 * DataModel} instance or used for the configuration of a built-in AuboScope
 * program node.
 */
class ARCS_ABI_EXPORT VariableModel
{
public:
    VariableModel(VariableModel &f);
    VariableModel(VariableModel &&f);
    ~VariableModel();

    /**
     *
     * @return the collection of all variables that currently are registered in
     * the system.
     */
    std::vector<VariablePtr> getAll();

    VariablePtr get(const std::string &name);

    /**
     * Get a subset of all the variables registered in the system using a
     * filter.
     *
     * @param filter see {@link VariableFilterFactory} for examples.
     * @return the collection of Variables that are accepted by the filter.
     */
    std::vector<VariablePtr> get(std::function<bool(VariablePtr)> filter =
                                     [](VariablePtr) { return true; });

    /**
     * @return a {@link VariableFactory} to create variable objects.
     */
    VariableFactoryPtr getVariableFactory();

    /**
     *
     * Get the feature variable for a {@link Feature}.
     *
     * @param feature the feature with a variable.
     * @return the variable for the feature.
     * @throws IllegalArgumentException If feature not is present in the
     * installation or feature has no variable, see {@link
     * Feature#isVariable()}.
     */
    FeatureVariablePtr getFeatureVariable(FeaturePtr feature);

    /**

     * Register callback for changed
     *
     * @param receiver
     * @param slot
     */
    std::string connectedTo(const std::string &receiver,
                            const std::function<void(VariableModel *)> &slot);

    void disconnectFrom(const std::string &receiver);

private:
    friend class DataSwitch;
    VariableModel();
    void *d_{ nullptr };
};

} // namespace aubo_scope
} // namespace arcs
#endif // AUBO_SCOPE_VARIABLE_MODEL_H
