#ifndef AUBO_SCOPE_WAYPOINT_MODEL_H
#define AUBO_SCOPE_WAYPOINT_MODEL_H

#include <vector>
#include <aubo_caps/domain/waypoint/waypoint.h>

namespace arcs {
namespace aubo_scope {
ARCS_CLASS_FORWARD(WaypointModel);

/**
 * Provides methods that returns Waypoints from the current robot installation.
 */
class ARCS_ABI_EXPORT WaypointModel
{
public:
    WaypointModel(WaypointModel &f);
    WaypointModel(WaypointModel &&f);
    ~WaypointModel();

    /**
     * @return the geometric Waypoints of the current installation
     */
    /**
     * @param clazz the sort of geometric Waypoint of interest, e.g.
     * <code>WaypointPoint.class</code>,
     *              <code>WaypointLine.class</code>,
     * <code>WaypointPlane.class</code>
     * @param <T> The Waypoint type
     * @return the collection of the corresponding Waypoints
     */
    std::vector<WaypointPtr> getWaypoints();

    WaypointPtr getByName(const std::string &name);

    /**
     * Add a Waypoint to the current AuboScope installation. This makes it
     * selectable by the end user. The Waypoint is not modifiable by the end
     * user.
     *
     * @param idKey The key to identify this Waypoint by. The key is for this
     * AuboCap only, i.e. it only has to be unique for this AuboCap and not
     * "globally" for other aubo_studio plugins.
     * @param suggestedName Suggested name for the Waypoint. Valid names must
     * match regex [a-zA-Z][a-zA-Z0-9_]{0,14} for a total of 15 characters. The
     * final name can be retrieved from the returned Waypoint instance.
     * @param pose The pose of the Waypoint with respect to the robot base
     * @param joint The joint of the Waypoint with respect to the robot base
     * @param offset The tcp of the Waypoint with respect to the robot base
     * @return The Waypoint created and registered in AuboScope.
     * @throws WaypointAlreadyAddedException If a Waypoint has previously been
     * added the same <code>idKey</code> identifier. Use {@link
     * #getWaypoint(String)} to check if the Waypoint has already been added to
     * the current installation. Use {@link #updateWaypoint(String, Pose)} to
     * update the Waypoint.
     * @throws IllegalWaypointNameException If the suggested name does not match
     * required regex.
     */
    WaypointPtr addWaypoint(const std::string &suggestedName,
                            const std::vector<double> &pose);

    WaypointPtr addWaypoint(const std::string &suggestedName,
                            const std::vector<double> &pose,
                            const std::vector<double> &joint,
                            const std::vector<double> &offset);
    /**
     * Remove a Waypoint added by this AuboCap from AuboScope. Program nodes
     * using the Waypoint will be become undefined because the Waypoint is no
     * longer resolvable.
     *
     * @param idKey The key used to add the Waypoint with.
     * @throws WaypointNotFoundException If no Waypoint exists with the provided
     * <code>idKey</code>.
     */
    void removeWaypoint(const std::string &name);

    /**
     * Rename the waypoint in the model by the name
     * @param name
     * @param newName
     */
    void renameWaypoint(const std::string &name, const std::string &newName);

private:
    friend class DataSwitch;
    WaypointModel();
    void *d_{ nullptr };
};

} // namespace aubo_scope
} // namespace arcs

#endif // AUBO_SCOPE_WAYPOINT_MODEL_H
