# **`MyFirstAuboCap`--开发`Service`类**

在本小结，我们将为`hello`程序节点开发`Service`类。

`Service`类 (ProgramNodeService) 的目的:

> `service`类提供了配置节点基本信息的函数、创建该节点的用户交互界面（`view`类）的函数以及创建每个节点的`Contribution`类的函数。

------

该类中对于各种重载函数的详细描述，可在文章[程序节点的集成原理](Principle_of_aubocaps_integration_in_aubo_scope.md)中查看。

1. 由于`ProgramNodeService`接口的实现需要同时引用节点的`view`类和`Contribution`类,因此首先需要创建节点的`view`类和`Contribution`类。
2. `ProgramNodeService`接口需要定义节点ID以及节点的标题，如可用程序节点列表中所示。此外，`ContributionConfiguration`允许修改节点参数集，例如节点是否可以具有子节点（父节点）。这些设置必须使用适当的值进行配置。
3. 该服务还引用链接到该节点的`View` 和`Contribution`。因此，服务应在调用`createView（)`时返回`View`类的实例，并在调用` createNode()`时返回`Contribution`的实例。这两个类在其构造函数中可能需要API对象，例如`DataModel`和`API Provider`。

------

**MyfirstAuboCap的文件结构如下：**

![](pics/sample_programNodeFile_structure.png)

------

**hello节点`service`类的示例代码如下：**

1. `hello_program_node_service.h`

   ```c++
   #ifndef HELLO_PROGRAM_NODE_SERVICE_H
   #define HELLO_PROGRAM_NODE_SERVICE_H
   
   #include <aubo_caps/contribution/program/program_node_service.h>
   #include "hello_program_node_contribution.h"
   #include "hello_program_node_view.h"
   
   using namespace arcs::aubo_scope;
   
   class helloProgramNodeService : public ProgramNodeService
   {
   public:
       helloProgramNodeService();
       ~helloProgramNodeService();
       std::string getId() override;
       std::string getTitle() override;
       std::string getIcon() override;
       void configureContribution(
               ProgramContributionConfigurationPtr configuration) override;
       ProgramNodeViewPtr createView(ViewApiProviderPtr api_provider) override;
       ProgramNodeContributionPtr createNode(
               ProgramApiProviderPtr api_provider, ProgramNodeViewPtr view,
               DataModelPtr model, ProgramCreationContextPtr context) override;
   };
   
   #endif // HELLO_PROGRAM_NODE_SERVICE_H
   ```

2. `hello_program_node_service.cpp`

   ```c++
   #include "hello_program_node_service.h"
   #include "hello_program_node_view.h"
   
   helloProgramNodeService::helloProgramNodeService(){}
   helloProgramNodeService::~helloProgramNodeService(){}
   
   std::string helloProgramNodeService::getId()
   {
       return "hello";
   }
   
   std::string helloProgramNodeService::getTitle()
   {
       return "hello";
   }
   
   std::string helloProgramNodeService::getIcon()
   {
       return "img:node-comment.png";
   }
   
   void helloProgramNodeService::configureContribution(
           ProgramContributionConfigurationPtr configuration)
   {
   }
   
   ProgramNodeViewPtr helloProgramNodeService::createView(
           ViewApiProviderPtr api_provider)
   {
       return std::make_shared<helloProgramNodeView>(api_provider);
   }
   
   ProgramNodeContributionPtr helloProgramNodeService::createNode(
           ProgramApiProviderPtr api_provider, ProgramNodeViewPtr view,
           DataModelPtr model, ProgramCreationContextPtr context)
   {
       return std::make_shared<helloProgramNodeContribution>(
                   api_provider, std::dynamic_pointer_cast<helloProgramNodeView>(view),
                   model);
   }
   ```

------

**在激活器中注册服务：**

当service类完成时，需要在激活器中注册此服务。激活器注册aubo_cap 应该提供给aubo_scope的所有服务。当aubo_scope启动时，将调用激活器start()方法，并且应在bundle context中注册相关服务。

`activator.cpp`

```c++
#include "cppmicroservices/Bundle.h"
#include "cppmicroservices/BundleActivator.h"
#include "cppmicroservices/BundleImport.h"
#include "cppmicroservices/BundleContext.h"
#include "cppmicroservices/BundleResource.h"
#include "cppmicroservices/BundleResourceStream.h"
#include "cppmicroservices/Constants.h"
#include "cppmicroservices/ServiceEvent.h"

#include "hello_program_node_service.h"

using namespace cppmicroservices;

/**
 * This class implements a simple bundle that utilizes the CppMicroServices's
 * event mechanism to listen for service events. Upon receiving a service event,
 * it prints out the event's details.
 */
class Activator : public BundleActivator
{
private:
    /**
     * Implements BundleActivator::Start(). Prints a message and adds a member
     * function to the bundle context as a service listener.
     *
     * @param context the framework context for the bundle.
     */
    void Start(BundleContext context)
    {
        auto bundle = context.GetBundle();
        auto headers = bundle.GetHeaders();

        cppmicroservices::ServiceProperties props;
        props["Name"] = std::string("MyFirstAuboCap");
        props["VendorName"] = std::string("aubo-robotics");
        props["Version"] = 0;

        //注册节点
        context.RegisterService<arcs::aubo_scope::ProgramNodeService>(
            std::make_shared<helloProgramNodeService>(), props);
    }

    /**
     * Implements BundleActivator::Stop(). Prints a message and removes the
     * member function from the bundle context as a service listener.
     *
     * @param context the framework context for the bundle.
     */
    void Stop(BundleContext /*context*/) {}
};

CPPMICROSERVICES_EXPORT_BUNDLE_ACTIVATOR(Activator)
```

------

